<?php
/**
 * Plugin Name: Google Places Reviews Pro
 * Plugin URI: https://wordimpress.com/plugins/google-places-reviews-pro/
 * Description: Display Google Places Reviews for one or many businesses anywhere on your WordPress site using an easy to use and intuitive shortcode and widget. This is the premium version of the plugin. Thank you for your purchase and supporting sustained development.
 * Version: 1.5.5
 * Author: WordImpress
 * Author URI: https://wordimpress.com/
 * Text Domain: google-places-reviews
 * Domain Path: /languages/
 */

if ( ! class_exists( 'WP_Google_Places_Reviews' ) ) {

	/**
	 * Class WP_Google_Places_Reviews
	 */
	class WP_Google_Places_Reviews {

		/**
		 * Plugin Instance
		 *
		 * @var null
		 */
		protected static $_instance = null;

		/**
		 * GPR_Plugin_Framework
		 *
		 * @var GPR_Plugin_Framework
		 */
		public $gpr_framework = null;


		/**
		 * Main Google Places Reviews Instance
		 *
		 * Ensures only one instance of GPR is loaded or can be loaded.
		 *
		 * @since 1.2.2
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}

			return self::$_instance;
		}

		/**
		 * Class Constructor
		 */
		public function __construct() {

			add_action( 'init', array( $this, 'load_plugin_textdomain' ) );

			$this->check_for_free_version();

		}

		/**
		 * Check for Free Version
		 *
		 * @since 1.5
		 */
		function check_for_free_version() {

			$active_plugins = (array) get_option( 'active_plugins', array() );

			//Is the free version activated?
			if ( in_array( 'google-places-reviews/google-places-reviews.php', $active_plugins ) ) {

				add_action( 'admin_init', array( $this, 'deactivate_plugin' ) );
				add_action( 'admin_notices', array( $this, 'free_version_admin_notice' ) );

			} else {

				//All good, proceed with activation
				$this->init();

			}

		}

		/**
		 * Initialize the Plugin
		 */
		function init() {

			$this->define_globals();

			//Register the Google Places widget
			add_action( 'widgets_init', array( $this, 'setup_widget' ) );

			//Plugin Row links
			add_filter( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 10, 2 );

			//Shortcode generator for TinyMCE
			require_once GPR_PLUGIN_PATH . '/classes/class-shortcode-generator.php';
		}

		/**
		 * Load Languages
		 */
		function load_plugin_textdomain() {

			// Set filter for plugins's languages directory
			$gpr_lang_dir = dirname( plugin_basename( __FILE__ ) ) . '/languages/';
			$gpr_lang_dir = apply_filters( 'gpr_languages_directory', $gpr_lang_dir );

			// Traditional WordPress plugin locale filter
			$locale = apply_filters( 'plugin_locale', get_locale(), 'google-places-reviews' );
			$mofile = sprintf( '%1$s-%2$s.mo', 'give', $locale );

			// Setup paths to current locale file
			$mofile_local  = $gpr_lang_dir . $mofile;
			$mofile_global = WP_LANG_DIR . '/google-places-reviews/' . $mofile;

			if ( file_exists( $mofile_global ) ) {
				// Look in global /wp-content/languages/google-places-reviews folder
				load_textdomain( 'google-places-reviews', $mofile_global );
			} elseif ( file_exists( $mofile_local ) ) {
				// Look in local location from filter `gpr_languages_directory`
				load_textdomain( 'google-places-reviews', $mofile_local );
			} else {
				// Load the default language files packaged up w/ Google Places Reviews
				load_plugin_textdomain( 'google-places-reviews', false, $gpr_lang_dir );
			}
		}

		/**
		 * Define Globals
		 */
		protected function define_globals() {

			//Define Constants
			$home_url    = home_url();
			$plugins_url = plugins_url();

			if ( stripos( $home_url, 'https://' ) === false ) {
				$plugins_url = str_ireplace( 'https://', 'http://', $plugins_url );
			}

			define( 'GPR_PLUGIN_NAME_PLUGIN', plugin_basename( __FILE__ ) );
			define( 'GPR_PLUGIN_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) );
			define( 'GPR_PLUGIN_URL', $plugins_url . '/' . basename( plugin_dir_path( __FILE__ ) ) );
		}


		/**
		 * Plugin Setup
		 */
		public function setup_widget() {

			add_action( 'admin_enqueue_scripts', array( $this, 'options_scripts' ) );

			// Include Core Framework class
			require_once GPR_PLUGIN_PATH . '/classes/core.php';

			require_once GPR_PLUGIN_PATH . '/inc/license/license.php';

			// Include Licensing
			if ( ! class_exists( 'EDD_SL_Plugin_Updater' ) ) {
				// load our custom updater
				require_once GPR_PLUGIN_PATH . '/inc/license/classes/EDD_SL_Plugin_Updater.php';
			}

			//Admin only
			if ( is_admin() ) {

				//Deactivating normal activation banner for upgrade to Place ID banner
				require_once GPR_PLUGIN_PATH . '/inc/activation-banner.php';

				//Display our upgrade notice
				require_once GPR_PLUGIN_PATH . '/inc/upgrades/upgrade-functions.php';
				require_once GPR_PLUGIN_PATH . '/inc/upgrades/upgrades.php';

			}

			// Create plugin instance
			$this->gpr_framework = new GPR_Plugin_Framework( __FILE__ );

			// Include options set
			include_once GPR_PLUGIN_PATH . '/inc/options.php';

			// Create options page
			$this->gpr_framework->add_options_page( array(), $google_places_reviews_options );

			//Include the widget
			if ( ! class_exists( 'Google_Places_Reviews' ) ) {
				require GPR_PLUGIN_PATH . '/classes/widget.php';
				require GPR_PLUGIN_PATH . '/classes/shortcode.php';
				new Google_Places_Reviews_Shortcode();
			}

			register_widget( 'Google_Places_Reviews' );

			return $this->gpr_framework;

		}


		/**
		 * Custom CSS for Options Page
		 */
		function options_scripts( $hook ) {
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			if ( $hook === 'settings_page_googleplacesreviews' ) {
				wp_register_style( 'gpr_custom_options_styles', plugin_dir_url( __FILE__ ) . '/dist/css/options' . $suffix . '.css' );
				wp_enqueue_style( 'gpr_custom_options_styles' );

			}
		}


		/**
		 * Free Version Activated
		 *
		 * @description: Displays and admin notice telling admin to deactivate free version
		 */
		public function free_version_admin_notice() {

			$error_message = __( '<div id="message" class="error"><p><strong>Activation Error:</strong> Please deactivate and uninstall the <em>free</em> version of Google Places Reviews and then reactivate the Pro version.</p> </div>', 'google-places-reviews' );

			echo $error_message;

			return false;
		}


		/**
		 * Deactivate Plugin
		 *
		 * @description: Displays and admin notice telling admin to deactivate free version
		 */
		public function deactivate_plugin() {

			//Deactivate this beast
			deactivate_plugins( plugin_basename( __FILE__ ) );

		}


		/**
		 * Get Version Number
		 *
		 * @description: Helper function to easily retrieve the version number
		 *
		 * @since      1.5
		 *
		 * @return string the version number
		 */
		public function get_version_number() {

			// Create plugin instance
			$framework = new GPR_Plugin_Framework( __FILE__ );

			return $framework->version;
		}

		/**
		 * Plugin row meta links
		 *
		 * @since 1.4
		 *
		 * @param array  $input already defined meta links
		 * @param string $file  plugin file path and name being processed
		 *
		 * @return array $input
		 */
		function plugin_row_meta( $input, $file ) {

			if ( $file != plugin_basename( __FILE__ ) ) {
				return $input;
			}

			$addons_link = esc_url( add_query_arg( array(
					'utm_source'   => 'plugins-page',
					'utm_medium'   => 'plugin-row',
					'utm_campaign' => 'admin',
				), 'https://wordimpress.com/support/' )
			);

			$docs_link = esc_url( add_query_arg( array(
					'utm_source'   => 'plugins-page',
					'utm_medium'   => 'plugin-row',
					'utm_campaign' => 'admin',
				), 'https://wordimpress.com/documentation/google-places-reviews/' )
			);

			$links = array(
				'<a href="' . $docs_link . '" target="_blank">' . esc_html__( 'View Documentation', 'give' ) . ' &raquo;</a>',
				'<a href="' . $addons_link . '" target="_blank">' . esc_html__( 'Get Support', 'give' ) . ' &raquo;</a>',
			);

			$input = array_merge( $input, $links );

			return $input;
		}


	}//end WP_Google_Places_Reviews class

}//end If !class_exists

/**
 * Returns the main instance of Google Places Reviews.
 *
 * @since  1.0
 * @return Google_Places_Reviews()
 */
function WP_Google_Places_Reviews() {
	return WP_Google_Places_Reviews::instance();
}

WP_Google_Places_Reviews();